/********************************************************************
*  document handlers for all the ptz control and screen buttons     *
*********************************************************************/


/***********************
 *  global variables   *
 ***********************/
let params = getParams(); // decode URL parameters
let selectedMem = 1; // default selected mem position
let canSave = false; // can save position
let saveDelay = 0; // delay movement counter before can save
let indrag = false; // set mouse drag state
let rx = 0; // relative x movement
let ry = 0; // relative y movement

/*******************************************
 * Initialize the ptz camera on page load  *
 *******************************************/
document.addEventListener('DOMContentLoaded', onLoad); // run onLoad function when DOM is ready

function onLoad() {
    document.querySelector("#video").setAttribute("src", "http://" + params["video"] + "/mjpg/video.mjpg"); // set video source
    addEventListeners(); // add all event listeners to the html elements
    setCamCmd(PTZ_HOME); // send ptz to home position
    setDragLine(0,0); // initialize drag line
    setSaveState(MEM_NO_SAVE); // set save state to idle
    setCloseState(CLOSE_BUTTON); // set close button
    setZoomStates(ZOOM_IN_IDLE, ZOOM_OUT_IDLE); // set zoom states to idle
    setMemStates(selectedMem, MEM_SELECTED, MEM_UNSELECTED); // set mem1 to selected, rest to unselected
    setInterval(updatePtzPosition, PTZ_CMD_DELAY); // update the ptz position every xxx milliseconds (can be changed in constants.js)
}

/***************************************
 * add all the screen event handlers   *
 ***************************************/
function addEventListeners() {
    const overlay = document.getElementById("overlay"); // add event listeners to the overlay div
    overlay.addEventListener('mousedown', handleMouseDown);
    overlay.addEventListener('mouseup', handleMouseUp);
    overlay.addEventListener('mousemove', handleMouseMove);
    overlay.addEventListener('mouseleave', handleMouseUp);

    const zoomin = document.getElementById("ptz_zoom_in"); // add event listeners to the zoom in button
    zoomin.addEventListener('mousedown', handleZoomInDown);
    zoomin.addEventListener('mouseup', handleZoomInUp);
    zoomin.addEventListener('mouseover', handleZoomInOver);
    zoomin.addEventListener('mouseleave', handleZoomInLeave);

    const zoomout = document.getElementById("ptz_zoom_out"); // add event listeners to the zoom out button
    zoomout.addEventListener('mousedown', handleZoomOutDown);
    zoomout.addEventListener('mouseup', handleZoomOutUp);
    zoomout.addEventListener('mouseover', handleZoomOutOver);
    zoomout.addEventListener('mouseleave', handleZoomOutLeave);

    for (let i = 1; i <= numberOfMemButtons; i++) { // add event listeners to the memory buttons
        const memButton = document.getElementById("ptz_mem" + i); // get the memory button and add event listeners
        memButton.addEventListener('mouseup', handleMemUp);
        memButton.addEventListener('mouseover', handleMemOver);
        memButton.addEventListener('mouseleave', handleMemLeave);
        memButton.addEventListener('mousedown', stopPropagation);
    }

    const close = document.getElementById("close_button"); // add event listeners to the close button
    close.addEventListener('mousedown', stopPropagation);
    close.addEventListener('mouseup', handleCloseUp);
    close.addEventListener('mouseover', handleCloseOver);
    close.addEventListener('mouseleave', handleCloseLeave);

    const ptzicon = document.getElementById("ptzicon"); // add event listeners to the ptz icon
    ptzicon.addEventListener('mouseup', handlePtzIconUp);


    const memsave = document.getElementById("ptz_mem_save"); // add event listeners to the memory save button
    memsave.addEventListener('mouseup', handleMemSaveUp);
    memsave.addEventListener('mousedown', stopPropagation);
}




/*********************************************
 *  event handlers for the main ptz overlay  *
 *********************************************/
function handleMouseDown(event) { // mouse down event
    stopPropagation(event);
    indrag = true; 
    saveDelay = 0; // reset delay counter
    canSave = false; 
    calcMovement(event);  // calculate movement for background loop
}

function handleMouseUp(event) { // mouse up event on overlay
    stopPropagation(event);
    indrag = false; 
    rx = 0; 
    ry = 0; 
    setDragLine(0,0); // clear drag line
    setCamCmd(PTZ_STOP_ALL); 
}

function handleMouseMove(event) { // mouse move event on overlay
    stopPropagation(event);
    if (indrag == true) { // are we in drag mode?
        calcMovement(event); // yes, calculate movement for background loop
    }
}

/********************************************
 *  event handlers for the memory buttoms   *
 ********************************************/
function handleMemUp(event) { // load position from selected memory in ptz camera
    stopPropagation(event);
    let mem = getMemId(event); 
    selectedMem = mem; // remember selected mem
    indrag = false;  
    canSave = false; 
    setMemStates(mem, MEM_SELECTED, MEM_UNSELECTED);  // set mem button to selected, rest to unselected
    setSaveState(MEM_NO_SAVE); 
    setDragLine(0,0); // clear drag line
    setCamCmd(PTZ_GOTO_MEM + mem); // send ptz camera to mem position
}

function handleMemOver(event) { // hover over memory button
    stopPropagation(event);
    let mem = getMemId(event); 
    if (mem == selectedMem) { // is this the selected memory button
        if (canSave) { // yes, we can save?
            setMemState(mem, MEM_RELOAD_HOVER); // yes
        } else {
            setMemState(mem, MEM_SELECTED); // no
        }
    } else {
        setMemState(mem, MEM_UNSELECTED_HOVER); // not selected
    }
}

function handleMemLeave(event) { // hover leave memory button
    stopPropagation(event);
    let mem = getMemId(event); 
    if (mem == selectedMem) { // is this the selected memory button
        if (canSave) { // yes, we can save?
            setMemState(mem, MEM_RELOAD); // yes
        } else {
            setMemState(mem, MEM_SELECTED); // no
        }
    } else {
        setMemState(mem, MEM_UNSELECTED); // not selected
    }
}

/****************************************
*  event handlers for the save buttom   *
*****************************************/
function handleMemSaveUp(event) { // save current position to selected memory in ptz camera
    stopPropagation(event);
    if (canSave) {
        canSave = false; // disable save until we move again
        setCamCmd(PTZ_MEM_SAVE + selectedMem); // save postion in camera pos selectedmem
        setSaveState(MEM_NO_SAVE); 
        setMemStates(selectedMem, MEM_SELECTED, MEM_UNSELECTED); 
    }
}

/****************************************
*  event handlers for the close buttom   *
*****************************************/
function handleCloseUp(event) { //  close button pressed
    stopPropagation(event);
    console.log("close");
    document.querySelector("#overlay").style.display = "none";
    document.querySelector("#close_button").style.display = "none";
    document.querySelector("#ptzicon").style.display = "inline";
}

function handleCloseLeave(event) { //  close button leave
    stopPropagation(event);
    setCloseState(CLOSE_BUTTON);
}

function handleCloseOver(event) { //  close button hoover
    stopPropagation(event);
    setCloseState(CLOSE_BUTTON_HOVER);
}

function handlePtzIconUp(event) { //  close button hoover
    stopPropagation(event);
    document.querySelector("#overlay").style.display = "inline";
    document.querySelector("#close_button").style.display = "inline";
    document.querySelector("#ptzicon").style.display = "none";
}



/***********************************************
 *  event handlers for the zoom buttoms        *
 ***********************************************/
function handleZoomInDown(event) { // zoom in button down
    stopPropagation(event);
    setCamCmd(PTZ_ZOOM_IN); 
    setZoomStates(ZOOM_IN_ACTIVE, ZOOM_OUT_IDLE); 
}

function handleZoomInOver(event) { // zoom in button hover
    stopPropagation(event);
    setZoomStates(ZOOM_IN_HOVER, ZOOM_OUT_IDLE); 
}

function handleZoomOutOver(event) { // zoom out button hover
    stopPropagation(event);
    setZoomStates(ZOOM_IN_IDLE, ZOOM_OUT_HOVER); 
}

function handleZoomInUp(event) { // zoom in button up
    stopPropagation(event);
    setCamCmd(PTZ_ZOOM_STOP);
    setZoomStates(ZOOM_IN_HOVER, ZOOM_OUT_IDLE); 
}

function handleZoomInLeave(event) { // zoom in button leave
    stopPropagation(event);
    setCamCmd(PTZ_ZOOM_STOP);
    setZoomStates(ZOOM_IN_IDLE, ZOOM_OUT_IDLE);
}

function handleZoomOutDown(event) { // zoom out button down
    stopPropagation(event);
    setCamCmd(PTZ_ZOOM_OUT); 
    setZoomStates(ZOOM_IN_IDLE, ZOOM_OUT_ACTIVE); // 
}

function handleZoomOutUp(event) { //
    stopPropagation(event);
    setCamCmd(PTZ_ZOOM_STOP);
    setZoomStates(ZOOM_IN_IDLE, ZOOM_OUT_HOVER);
}

function handleZoomOutLeave(event) { // zoom out button leave
    stopPropagation(event);
    setCamCmd(PTZ_ZOOM_STOP);
    setZoomStates(ZOOM_IN_IDLE, ZOOM_OUT_IDLE); 
}
